###############################################################################
# Copyright
# The IEEE Standards publication(s) ("Document") is approved by the
# IEEE Standards Association ("IEEE-SA") Standards Board and is published in
# accordance with established IEEE-SA Standards Board bylaws and operations
# procedures.
#
# IEEE owns the copyright to this Document in all forms of media. Copyright in
# the text retrieved, displayed or output from this Document is owned by IEEE
# and is protected by the copyright laws of the United States and by
# international treaties. IEEE reserves all rights not expressly granted.
#
# IEEE is providing the Document to you at no charge. However, the Document is
# not to be considered within the "Public Domain", as IEEE is, and at all times
# shall remain, the sole copyright holder in the Document.
#
# Terms of Use
# You may retrieve, download and print one (1) copy of each Document in this
# Program for your personal use. You may retain one (1) additional copy of this
# Document as your personal archive copy.
#
# Except as allowed by the copyright laws of the United States of America or
# applicable international treaties, or as explicitly allowed in these Terms of
# Use, you may not further copy, prepare, and/or distribute copies of the
# Document, nor significant portions of the Document, in any form, without
# prior written permission from IEEE.
#
# Requests for permission to reprint this Document, in whole or in part, or
# requests for a license to reproduce and/or distribute this Document, in any
# form, must be submitted via email to the Standards Licensing and Contracts
# (http://standards.ieee.org/contact/form.html), or in writing to:
#
# IEEE-SA Licensing and Contracts 
# 445 Hoes Lane 
# Piscataway, NJ 08855
###############################################################################
#
# Python script to post-process IEEE 2030.5 XSD

from lxml import etree

XSDNAMESPACE = '{http://www.w3.org/2001/XMLSchema}'
WADLNAMESPACE = '{http://wadl.dev.java.net/2009/02}'

def add_default_to_element(root, type_name, element_name, default_value):
    default_element = root.find(XSDNAMESPACE + "complexType[@name='" + type_name + "']/" +
                                XSDNAMESPACE + "complexContent/" +
                                XSDNAMESPACE + "extension/" +
                                XSDNAMESPACE + "sequence/" +
                                XSDNAMESPACE + "element[@name='" + element_name + "']"
                               )
    default_element.attrib['default'] = default_value

wadl_elements = set()

tree = etree.parse('sep_wadl.xml')
root = tree.getroot()

for resources in root.iterchildren(WADLNAMESPACE + 'resources'):
    for resource in resources.iterchildren(WADLNAMESPACE + 'resource'):
        for method in resource.iterchildren(WADLNAMESPACE + 'method'):
            for element in method.iterchildren():
                for representation in element.iterchildren(WADLNAMESPACE + 'representation'):
                    wadl_elements.add(representation.attrib.get('element')[4:])  # remove first characters (sep:)

print(str(len(wadl_elements)) + ' elements in WADL')

tree = etree.parse('sep.xsd')
root = tree.getroot()

# The consensus of the editors ad hoc was:
# - Keep simpleTypes as simpleTypes and do not add any extensibility elements (for elements or attributes).
# - For complexTypes with simpleContent, do not add any extensibility elements for elements, but do add extensibility element for attributes (anyAttribute).
# - For complexTypes that do not contain simpleContent or complexContent add all of the extensibility elements (for elements and attributes).
# - For any complexType with complexContent that extends a complexType with simpleContent, do not add the extensibility elements related to elements, but do add the extensibility element related to attributes (anyAttribute).
# As a result of schema validation issues discovered during the -2023 Corrigendum 1 process, the following rules were added:
# - In addition to the above rules, only add the extensibility element for non-2030.5 elements (##other namespace) to base classes.
# - Prepend the type name to the extensibility element for 2030.5 elements (r2_3).

# Create a list of all types that are base types (i.e., do not extend other types)
base_types = []
for item in root.iterchildren():
    if item.tag == XSDNAMESPACE + 'element':
        continue
    if item.tag == XSDNAMESPACE + 'annotation':
        continue
    simple_content = item.find(XSDNAMESPACE + 'simpleContent')
    if simple_content is not None and simple_content.find(XSDNAMESPACE + 'extension') is not None:
        continue
    complex_content = item.find(XSDNAMESPACE + 'complexContent')
    if complex_content is not None and complex_content.find(XSDNAMESPACE + 'extension') is not None:
        continue
    base_types.append(item.get('name'))

# Find all complexTypes with complexContent that extend complexTypes with simpleContent (as well as any that extend those and on and on...)
simple_content_list = [x.get('name') for x in root.iterchildren(XSDNAMESPACE + 'complexType') if x.find(XSDNAMESPACE + 'simpleContent') is not None]
extends_simple_content = []
search_list = simple_content_list
while True:
    new_search_list = []
    for complex_type in root.iterchildren(XSDNAMESPACE + 'complexType'):
        complex_content = complex_type.find(XSDNAMESPACE + 'complexContent')
        if complex_content is not None:
            # contains complexContent

            cc_next = None
            if complex_content.find(XSDNAMESPACE + 'extension') is not None:
                cc_next = complex_content.find(XSDNAMESPACE + 'extension')
            elif complex_content.find(XSDNAMESPACE + 'restriction') is not None:
                cc_next = complex_content.find(XSDNAMESPACE + 'restriction')

            if cc_next is None:
                raise Exception('complexContent does not include extension or restriction: ' + complex_type.get('name'))

            if cc_next.get('base') in search_list:
                extends_simple_content.append(complex_type.get('name'))
                new_search_list.append(complex_type.get('name'))

    if not new_search_list:
        break
    else:
        search_list = new_search_list.copy()

# Select all complexTypes
for complex_type in root.iterchildren(XSDNAMESPACE + 'complexType'):
    # complexType can contain different elements at the top:
    # - complexContent
    # - simpleContent
    # - sequence
    print("%s - %s" % (complex_type.tag, complex_type.get('name')))

    complex_content = complex_type.find(XSDNAMESPACE + 'complexContent')
    simple_content = complex_type.find(XSDNAMESPACE + 'simpleContent')
    sequence = complex_type.find(XSDNAMESPACE + 'sequence')

    # Add extension elements for future revisions and third party extensions
    if simple_content is None:
        # contains complexContent, sequence, or nothing

        if complex_content is not None:
            # contains complexContent

            cc_next = None
            if complex_content.find(XSDNAMESPACE + 'extension') is not None:
                # All IEEE 2030.5 complexContent currently contains extension
                cc_next = complex_content.find(XSDNAMESPACE + 'extension')
            elif complex_content.find(XSDNAMESPACE + 'restriction') is not None:
                cc_next = complex_content.find(XSDNAMESPACE + 'restriction')

            if cc_next is None:
                raise Exception('complexContent does not include extension or restriction: ' + complex_type.get('name'))

            sequence = cc_next.find(XSDNAMESPACE + 'sequence')

            if sequence is None and complex_type.get('name') not in extends_simple_content:
                # Add sequence to those complexTypes that simply extend but do not add anything (e.g., FlowReservationResponseResponse)
                sequence = etree.Element(XSDNAMESPACE + 'sequence')
                if len(cc_next) and cc_next[0].tag == XSDNAMESPACE + 'annotation':
                    # Insert after annotation
                    cc_next.insert(1, sequence)
                else:
                    # Insert at beginning
                    cc_next.insert(0, sequence)
                etree.indent(cc_next, level=3)

            # Add extension element for attributes (anyAttribute)
            anyAttr = etree.SubElement(cc_next, XSDNAMESPACE + 'anyAttribute')
            anyAttr.set('processContents', 'lax')
            etree.indent(cc_next, level=3)
        else:
            # does not contain complexContent

            if sequence is None and complex_type.get('name') not in extends_simple_content:
                # Add sequence
                sequence = etree.Element(XSDNAMESPACE + 'sequence')
                if len(complex_type) and complex_type[0].tag == XSDNAMESPACE + 'annotation':
                    # Insert after annotation
                    complex_type.insert(1, sequence)
                else:
                    # Insert at beginning
                    complex_type.insert(0, sequence)
                etree.indent(complex_type, level=2)

            # Add extension element for attributes (anyAttribute) directly to complex_type
            anyAttr = etree.SubElement(complex_type, XSDNAMESPACE + 'anyAttribute')
            anyAttr.set('processContents', 'lax')
            etree.indent(complex_type, level=2)

        if complex_type.get('name') not in extends_simple_content:
            # Add extension element for future revision (version 2.3)
            revType = etree.SubElement(sequence, XSDNAMESPACE + 'element')
            revType.set('name', complex_type.get('name') + '_r2_3')
            revType.set('type', 'Revision2_3Type')
            revType.set('minOccurs', '0')
            revType.set('maxOccurs', '1')

            if complex_type.get('name') in base_types:
                # Add extension element for third party extensions
                anyExt = etree.SubElement(sequence, XSDNAMESPACE + 'any')
                anyExt.set('namespace', '##other')
                anyExt.set('processContents', 'lax')
                anyExt.set('minOccurs', '0')
                anyExt.set('maxOccurs', 'unbounded')
                etree.indent(sequence, level=4)
    else:
        # contains simpleContent

        sc_next = None
        if simple_content.find(XSDNAMESPACE + 'extension') is not None:
            # All IEEE 2030.5 simpleContent currently contains extension
            sc_next = simple_content.find(XSDNAMESPACE + 'extension')
        elif simple_content.find(XSDNAMESPACE + 'restriction') is not None:
            sc_next = simple_content.find(XSDNAMESPACE + 'restriction')

        if sc_next is None:
            raise Exception('simpleContent does not include extension or restriction: ' + complex_type.get('name'))

        # Add extension element for attributes (anyAttribute)
        anyAttr = etree.SubElement(sc_next, XSDNAMESPACE + 'anyAttribute')
        anyAttr.set('processContents', 'lax')
        etree.indent(sc_next, level=3)

# Add Revision2_3Type
revType = etree.SubElement(root, XSDNAMESPACE + 'complexType')
revType.set('name', 'Revision2_3Type')
sequence = etree.SubElement(revType, XSDNAMESPACE + 'sequence')
anyExt = etree.SubElement(sequence, XSDNAMESPACE + 'any')
anyExt.set('processContents', 'lax')
anyExt.set('minOccurs', '1')
anyExt.set('maxOccurs', 'unbounded')
anyExt.set('namespace', '##targetNamespace')
anyAtt = etree.SubElement(revType, XSDNAMESPACE + 'anyAttribute')
anyAtt.set('processContents', 'lax')
etree.indent(root, level=0)

# Select all top-level elements
for element in root.iterchildren(XSDNAMESPACE + 'element'):
    print("%s - %s" % (element.tag, element.get('name')))

    if element.get('name') not in wadl_elements:
        # Remove top-level elements that are not used in the top level (e.g., data types)
        root.remove(element)
    else:
        # Add schemaVer attribute to top-level elements
        element.attrib.pop('type')
        ct = etree.SubElement(element, XSDNAMESPACE + 'complexType')
        complex_content = etree.SubElement(ct, XSDNAMESPACE + 'complexContent')
        ex = etree.SubElement(complex_content, XSDNAMESPACE + 'extension')
        ex.set('base', element.get('name'))
        at = etree.SubElement(ex, XSDNAMESPACE + 'attribute')
        at.set('name', 'schemaVer')
        at.set('use', 'optional')
        at.set('default', '2.1')
        at.set('type', 'SEPVersion')
        an = etree.SubElement(at, XSDNAMESPACE + 'annotation')
        dc = etree.SubElement(an, XSDNAMESPACE + 'documentation')
        dc.text = 'The schema version used. All XML payloads SHALL include a schemaVer attribute in the top-level XML element equal to the version of the IEEE 2030.5 schema (IEEE Std 2030.5 supplemental material) used (e.g., schemaVer="2.2"). It should be noted that previous revisions of IEEE 2030.5 did not require this schemaVer attribute.'
        etree.indent(element, level=1)

# Add default values to elements that have defaults. Note: this is necessary as
# CIMEA currently adds the defaults for attributes but not for elements.
add_default_to_element(root, 'DeviceStatus', 'opState', '0')
add_default_to_element(root, 'ExternalDevice', 'enabled', 'true')
add_default_to_element(root, 'FunctionSetAssignments', 'version', '0')
add_default_to_element(root, 'DemandResponseProgram', 'availabilityUpdatePercentChangeThreshold', '0')
add_default_to_element(root, 'EndDeviceControl', 'overrideDuration', '0')
add_default_to_element(root, 'ReadingType', 'accumulationBehaviour', '0')
add_default_to_element(root, 'ReadingType', 'commodity', '0')
add_default_to_element(root, 'ReadingType', 'dataQualifier', '0')
add_default_to_element(root, 'ReadingType', 'flowDirection', '0')
add_default_to_element(root, 'ReadingType', 'kind', '0')
add_default_to_element(root, 'ReadingType', 'numberOfConsumptionBlocks', '0')
add_default_to_element(root, 'ReadingType', 'numberOfTouTiers', '0')
add_default_to_element(root, 'ReadingType', 'phase', '0')
add_default_to_element(root, 'ReadingType', 'powerOfTenMultiplier', '0')
add_default_to_element(root, 'ReadingType', 'tieredConsumptionBlocks', 'false')
add_default_to_element(root, 'ReadingType', 'uom', '0')
add_default_to_element(root, 'DERCurve', 'autonomousVRefEnable', 'false')
add_default_to_element(root, 'IdentifiedObject', 'version', '0')
add_default_to_element(root, 'RespondableIdentifiedObject', 'version', '0')
add_default_to_element(root, 'RespondableSubscribableIdentifiedObject', 'version', '0')
add_default_to_element(root, 'SubscribableIdentifiedObject', 'version', '0')
add_default_to_element(root, 'RandomizableEvent', 'randomizeDuration', '0')
add_default_to_element(root, 'RandomizableEvent', 'randomizeStart', '0')
add_default_to_element(root, 'ReadingBase', 'consumptionBlock', '0')
add_default_to_element(root, 'ReadingBase', 'qualityFlags', '00')
add_default_to_element(root, 'ReadingBase', 'touTier', '0')

etree.ElementTree(root).write('testout.xml', pretty_print=False, encoding='UTF-8', xml_declaration=True)

